/**
 * BotFake Recorder - Client-side action recording
 * Can be used as a bookmarklet or browser extension content script
 * 
 * Records: clicks, inputs, form submissions, navigation, scrolls
 * Outputs: JSON scenario compatible with BotFake test runner
 */

(function() {
  'use strict';

  // Prevent multiple injections
  if (window.__BOTFAKE_RECORDER__) {
    window.__BOTFAKE_RECORDER__.toggle();
    return;
  }

  const BotFakeRecorder = {
    isRecording: false,
    isPaused: false,
    actions: [],
    startTime: null,
    startUrl: null,
    
    // UI Elements
    panel: null,
    
    // Configuration
    config: {
      ignoreSelectors: ['#botfake-recorder-panel', '.botfake-recorder-highlight'],
      maxSelectorLength: 200
    },

    init() {
      this.createPanel();
      this.startUrl = window.location.href;
      window.__BOTFAKE_RECORDER__ = this;
      console.log('[BotFake Recorder] Initialized');
    },

    toggle() {
      if (this.panel.style.display === 'none') {
        this.panel.style.display = 'block';
      } else {
        this.panel.style.display = 'none';
      }
    },

    createPanel() {
      // Create floating panel
      this.panel = document.createElement('div');
      this.panel.id = 'botfake-recorder-panel';
      this.panel.innerHTML = `
        <style>
          #botfake-recorder-panel {
            position: fixed;
            top: 10px;
            right: 10px;
            width: 320px;
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
            border: 2px solid #00d4ff;
            border-radius: 12px;
            box-shadow: 0 8px 32px rgba(0, 212, 255, 0.3);
            z-index: 2147483647;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            color: #fff;
            font-size: 14px;
          }
          #botfake-recorder-panel * {
            box-sizing: border-box;
          }
          .bfr-header {
            padding: 12px 16px;
            background: rgba(0, 212, 255, 0.1);
            border-bottom: 1px solid rgba(0, 212, 255, 0.3);
            display: flex;
            align-items: center;
            justify-content: space-between;
            border-radius: 10px 10px 0 0;
          }
          .bfr-logo {
            font-weight: 700;
            font-size: 16px;
            color: #00d4ff;
          }
          .bfr-close {
            background: none;
            border: none;
            color: #888;
            font-size: 20px;
            cursor: pointer;
            padding: 0;
            line-height: 1;
          }
          .bfr-close:hover { color: #ff4757; }
          .bfr-body {
            padding: 16px;
          }
          .bfr-status {
            display: flex;
            align-items: center;
            gap: 8px;
            margin-bottom: 12px;
            padding: 8px 12px;
            background: rgba(255,255,255,0.05);
            border-radius: 8px;
          }
          .bfr-status-dot {
            width: 10px;
            height: 10px;
            border-radius: 50%;
            background: #888;
          }
          .bfr-status-dot.recording {
            background: #ff4757;
            animation: bfr-pulse 1s infinite;
          }
          .bfr-status-dot.paused {
            background: #ffa502;
          }
          @keyframes bfr-pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.5; }
          }
          .bfr-actions-count {
            margin-left: auto;
            font-size: 12px;
            color: #888;
          }
          .bfr-buttons {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 8px;
            margin-bottom: 12px;
          }
          .bfr-btn {
            padding: 10px 16px;
            border: none;
            border-radius: 8px;
            font-size: 13px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.2s;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 6px;
          }
          .bfr-btn-primary {
            background: linear-gradient(135deg, #00d4ff 0%, #0099cc 100%);
            color: #000;
          }
          .bfr-btn-primary:hover { transform: translateY(-1px); box-shadow: 0 4px 12px rgba(0, 212, 255, 0.4); }
          .bfr-btn-secondary {
            background: rgba(255,255,255,0.1);
            color: #fff;
            border: 1px solid rgba(255,255,255,0.2);
          }
          .bfr-btn-secondary:hover { background: rgba(255,255,255,0.15); }
          .bfr-btn-danger {
            background: linear-gradient(135deg, #ff4757 0%, #cc0033 100%);
            color: #fff;
          }
          .bfr-btn-danger:hover { transform: translateY(-1px); box-shadow: 0 4px 12px rgba(255, 71, 87, 0.4); }
          .bfr-btn-success {
            background: linear-gradient(135deg, #2ed573 0%, #1e9c50 100%);
            color: #fff;
          }
          .bfr-btn-success:hover { transform: translateY(-1px); box-shadow: 0 4px 12px rgba(46, 213, 115, 0.4); }
          .bfr-btn:disabled {
            opacity: 0.5;
            cursor: not-allowed;
            transform: none !important;
          }
          .bfr-btn-full {
            grid-column: span 2;
          }
          .bfr-actions-list {
            max-height: 200px;
            overflow-y: auto;
            background: rgba(0,0,0,0.3);
            border-radius: 8px;
            padding: 8px;
            font-size: 12px;
            font-family: 'Monaco', 'Menlo', monospace;
          }
          .bfr-action-item {
            padding: 4px 8px;
            margin-bottom: 4px;
            background: rgba(255,255,255,0.05);
            border-radius: 4px;
            border-left: 3px solid #00d4ff;
          }
          .bfr-action-item:last-child { margin-bottom: 0; }
          .bfr-action-type {
            color: #00d4ff;
            font-weight: 600;
          }
          .bfr-footer {
            padding: 12px 16px;
            border-top: 1px solid rgba(255,255,255,0.1);
            font-size: 11px;
            color: #666;
            text-align: center;
          }
          .bfr-footer a {
            color: #00d4ff;
            text-decoration: none;
          }
          .bfr-minimized .bfr-body,
          .bfr-minimized .bfr-footer { display: none; }
          .bfr-minimized { width: auto; }
        </style>
        <div class="bfr-header">
          <span class="bfr-logo">🎬 BotFake Recorder</span>
          <button class="bfr-close" id="bfr-close" title="Minimize">−</button>
        </div>
        <div class="bfr-body">
          <div class="bfr-status">
            <span class="bfr-status-dot" id="bfr-status-dot"></span>
            <span id="bfr-status-text">Ready to record</span>
            <span class="bfr-actions-count" id="bfr-actions-count">0 actions</span>
          </div>
          <div class="bfr-buttons">
            <button class="bfr-btn bfr-btn-primary bfr-btn-full" id="bfr-start">
              ⏺ Start Recording
            </button>
            <button class="bfr-btn bfr-btn-secondary" id="bfr-pause" disabled>
              ⏸ Pause
            </button>
            <button class="bfr-btn bfr-btn-danger" id="bfr-stop" disabled>
              ⏹ Stop
            </button>
            <button class="bfr-btn bfr-btn-success bfr-btn-full" id="bfr-download" disabled>
              📥 Download JSON
            </button>
            <button class="bfr-btn bfr-btn-secondary bfr-btn-full" id="bfr-copy" disabled>
              📋 Copy to Clipboard
            </button>
          </div>
          <div class="bfr-actions-list" id="bfr-actions-list">
            <div style="color: #666; text-align: center; padding: 20px;">
              Click "Start Recording" and perform actions on the page
            </div>
          </div>
        </div>
        <div class="bfr-footer">
          Powered by <a href="https://botfake.ru" target="_blank">BotFake.ru</a>
        </div>
      `;
      
      document.body.appendChild(this.panel);
      this.bindPanelEvents();
    },

    bindPanelEvents() {
      document.getElementById('bfr-close').addEventListener('click', () => {
        this.panel.classList.toggle('bfr-minimized');
        document.getElementById('bfr-close').textContent = 
          this.panel.classList.contains('bfr-minimized') ? '+' : '−';
      });
      
      document.getElementById('bfr-start').addEventListener('click', () => this.start());
      document.getElementById('bfr-pause').addEventListener('click', () => this.togglePause());
      document.getElementById('bfr-stop').addEventListener('click', () => this.stop());
      document.getElementById('bfr-download').addEventListener('click', () => this.download());
      document.getElementById('bfr-copy').addEventListener('click', () => this.copyToClipboard());
    },

    start() {
      this.isRecording = true;
      this.isPaused = false;
      this.actions = [];
      this.startTime = Date.now();
      this.startUrl = window.location.href;
      
      // Add initial navigation step
      this.addAction('web_navigate', { url: this.startUrl });
      
      this.bindRecordingEvents();
      this.updateUI();
      
      console.log('[BotFake Recorder] Recording started');
    },

    stop() {
      this.isRecording = false;
      this.isPaused = false;
      this.unbindRecordingEvents();
      this.updateUI();
      
      console.log('[BotFake Recorder] Recording stopped, ' + this.actions.length + ' actions recorded');
    },

    togglePause() {
      this.isPaused = !this.isPaused;
      this.updateUI();
    },

    updateUI() {
      const dot = document.getElementById('bfr-status-dot');
      const text = document.getElementById('bfr-status-text');
      const count = document.getElementById('bfr-actions-count');
      const startBtn = document.getElementById('bfr-start');
      const pauseBtn = document.getElementById('bfr-pause');
      const stopBtn = document.getElementById('bfr-stop');
      const downloadBtn = document.getElementById('bfr-download');
      const copyBtn = document.getElementById('bfr-copy');
      
      dot.className = 'bfr-status-dot';
      
      if (this.isRecording) {
        if (this.isPaused) {
          dot.classList.add('paused');
          text.textContent = 'Paused';
          pauseBtn.innerHTML = '▶ Resume';
        } else {
          dot.classList.add('recording');
          text.textContent = 'Recording...';
          pauseBtn.innerHTML = '⏸ Pause';
        }
        startBtn.disabled = true;
        pauseBtn.disabled = false;
        stopBtn.disabled = false;
        downloadBtn.disabled = true;
        copyBtn.disabled = true;
      } else {
        text.textContent = this.actions.length > 0 ? 'Stopped' : 'Ready to record';
        startBtn.disabled = false;
        startBtn.innerHTML = this.actions.length > 0 ? '⏺ New Recording' : '⏺ Start Recording';
        pauseBtn.disabled = true;
        stopBtn.disabled = true;
        downloadBtn.disabled = this.actions.length === 0;
        copyBtn.disabled = this.actions.length === 0;
      }
      
      count.textContent = this.actions.length + ' action' + (this.actions.length !== 1 ? 's' : '');
      this.renderActionsList();
      
      // Notify extension about update
      window.postMessage({
        type: 'BOTFAKE_RECORDER_UPDATE',
        actionsCount: this.actions.length,
        isRecording: this.isRecording
      }, '*');
    },

    renderActionsList() {
      const list = document.getElementById('bfr-actions-list');
      
      if (this.actions.length === 0) {
        list.innerHTML = `<div style="color: #666; text-align: center; padding: 20px;">
          Click "Start Recording" and perform actions on the page
        </div>`;
        return;
      }
      
      list.innerHTML = this.actions.map((action, i) => {
        const actionType = action.action.replace('web_', '');
        const detail = action.data.selector || action.data.url || action.data.value || '';
        const shortDetail = detail.length > 40 ? detail.substring(0, 40) + '...' : detail;
        return `<div class="bfr-action-item">
          <span class="bfr-action-type">${i + 1}. ${actionType}</span>
          ${shortDetail ? `<br><span style="color:#888">${this.escapeHtml(shortDetail)}</span>` : ''}
        </div>`;
      }).join('');
      
      list.scrollTop = list.scrollHeight;
    },

    escapeHtml(text) {
      const div = document.createElement('div');
      div.textContent = text;
      return div.innerHTML;
    },

    // Event handlers for recording
    handlers: {},

    bindRecordingEvents() {
      // Click handler
      this.handlers.click = (e) => {
        if (this.isPaused || this.isIgnored(e.target)) return;
        
        const selector = this.getSelector(e.target);
        const tagName = e.target.tagName.toLowerCase();
        
        // Don't record clicks on input fields (they trigger focus)
        if (['input', 'textarea', 'select'].includes(tagName)) return;
        
        this.addAction('web_click', { selector });
      };

      // Input handler (debounced)
      this.handlers.input = this.debounce((e) => {
        if (this.isPaused || this.isIgnored(e.target)) return;
        
        const selector = this.getSelector(e.target);
        const value = e.target.value;
        const tagName = e.target.tagName.toLowerCase();
        
        if (tagName === 'input' || tagName === 'textarea') {
          // Check if last action was same input - update value instead
          const lastAction = this.actions[this.actions.length - 1];
          if (lastAction && lastAction.action === 'web_fill' && lastAction.data.selector === selector) {
            lastAction.data.value = value;
            this.renderActionsList();
          } else {
            this.addAction('web_fill', { selector, value });
          }
        }
      }, 500);

      // Change handler for selects
      this.handlers.change = (e) => {
        if (this.isPaused || this.isIgnored(e.target)) return;
        
        if (e.target.tagName.toLowerCase() === 'select') {
          const selector = this.getSelector(e.target);
          const value = e.target.value;
          this.addAction('web_select', { selector, value });
        }
      };

      // Form submit
      this.handlers.submit = (e) => {
        if (this.isPaused) return;
        
        const selector = this.getSelector(e.target);
        this.addAction('web_submit', { selector });
      };

      // Keyboard (Enter, Escape)
      this.handlers.keydown = (e) => {
        if (this.isPaused || this.isIgnored(e.target)) return;
        
        if (e.key === 'Enter' && !['textarea'].includes(e.target.tagName.toLowerCase())) {
          const selector = this.getSelector(e.target);
          this.addAction('web_press', { selector, key: 'Enter' });
        }
      };

      // Scroll (debounced)
      this.handlers.scroll = this.debounce(() => {
        if (this.isPaused) return;
        
        const scrollY = window.scrollY;
        // Only record significant scrolls
        const lastAction = this.actions[this.actions.length - 1];
        if (lastAction && lastAction.action === 'web_scroll') {
          lastAction.data.y = scrollY;
          this.renderActionsList();
        } else if (scrollY > 100) {
          this.addAction('web_scroll', { y: scrollY });
        }
      }, 300);

      // Bind all handlers
      document.addEventListener('click', this.handlers.click, true);
      document.addEventListener('input', this.handlers.input, true);
      document.addEventListener('change', this.handlers.change, true);
      document.addEventListener('submit', this.handlers.submit, true);
      document.addEventListener('keydown', this.handlers.keydown, true);
      window.addEventListener('scroll', this.handlers.scroll, true);
    },

    unbindRecordingEvents() {
      document.removeEventListener('click', this.handlers.click, true);
      document.removeEventListener('input', this.handlers.input, true);
      document.removeEventListener('change', this.handlers.change, true);
      document.removeEventListener('submit', this.handlers.submit, true);
      document.removeEventListener('keydown', this.handlers.keydown, true);
      window.removeEventListener('scroll', this.handlers.scroll, true);
    },

    isIgnored(element) {
      return this.config.ignoreSelectors.some(sel => element.closest(sel));
    },

    addAction(action, data) {
      const step = {
        id: `step-${this.actions.length + 1}`,
        action: action,
        data: data,
        screenshot: false,
        timestamp: Date.now() - this.startTime
      };
      
      this.actions.push(step);
      this.updateUI();
      
      console.log('[BotFake Recorder] Action:', action, data);
    },

    getSelector(element) {
      // Try ID first
      if (element.id && !element.id.includes('bfr-')) {
        return `#${element.id}`;
      }
      
      // Try data-testid
      if (element.dataset.testid) {
        return `[data-testid="${element.dataset.testid}"]`;
      }
      
      // Try name attribute for form elements
      if (element.name) {
        return `[name="${element.name}"]`;
      }
      
      // Try unique class combination
      if (element.classList.length > 0) {
        const classes = Array.from(element.classList)
          .filter(c => !c.includes('hover') && !c.includes('active') && !c.includes('focus'))
          .slice(0, 2)
          .join('.');
        if (classes) {
          const selector = element.tagName.toLowerCase() + '.' + classes;
          if (document.querySelectorAll(selector).length === 1) {
            return selector;
          }
        }
      }
      
      // Build path selector
      const path = [];
      let current = element;
      
      while (current && current !== document.body && path.length < 5) {
        let selector = current.tagName.toLowerCase();
        
        if (current.id && !current.id.includes('bfr-')) {
          selector = `#${current.id}`;
          path.unshift(selector);
          break;
        }
        
        // Add nth-child if needed
        const parent = current.parentElement;
        if (parent) {
          const siblings = Array.from(parent.children).filter(c => c.tagName === current.tagName);
          if (siblings.length > 1) {
            const index = siblings.indexOf(current) + 1;
            selector += `:nth-of-type(${index})`;
          }
        }
        
        path.unshift(selector);
        current = current.parentElement;
      }
      
      return path.join(' > ').substring(0, this.config.maxSelectorLength);
    },

    generateScenario() {
      return {
        id: 'recorded-' + Date.now(),
        name: 'Recorded Scenario - ' + new Date().toLocaleString(),
        description: 'Recorded from ' + this.startUrl,
        startUrl: this.startUrl,
        type: 'website',
        browserType: 'chromium',
        headless: true,
        timeout: 60000,
        steps: this.actions.map(a => ({
          id: a.id,
          action: a.action,
          data: a.data,
          screenshot: a.screenshot
        }))
      };
    },

    download() {
      const scenario = this.generateScenario();
      const json = JSON.stringify(scenario, null, 2);
      const blob = new Blob([json], { type: 'application/json' });
      const url = URL.createObjectURL(blob);
      
      const a = document.createElement('a');
      a.href = url;
      a.download = `botfake-scenario-${Date.now()}.json`;
      a.click();
      
      URL.revokeObjectURL(url);
      console.log('[BotFake Recorder] Scenario downloaded');
    },

    copyToClipboard() {
      const scenario = this.generateScenario();
      const json = JSON.stringify(scenario, null, 2);
      
      navigator.clipboard.writeText(json).then(() => {
        const copyBtn = document.getElementById('bfr-copy');
        const originalText = copyBtn.innerHTML;
        copyBtn.innerHTML = '✓ Copied!';
        setTimeout(() => {
          copyBtn.innerHTML = originalText;
        }, 2000);
      }).catch(err => {
        console.error('[BotFake Recorder] Copy failed:', err);
        // Fallback
        const textarea = document.createElement('textarea');
        textarea.value = json;
        document.body.appendChild(textarea);
        textarea.select();
        document.execCommand('copy');
        document.body.removeChild(textarea);
      });
    },

    debounce(func, wait) {
      let timeout;
      return function(...args) {
        clearTimeout(timeout);
        timeout = setTimeout(() => func.apply(this, args), wait);
      };
    }
  };

  // Initialize
  BotFakeRecorder.init();
})();

